[CmdletBinding()]
param(
	# Name of the container. If a container with this name already exists, it is replaced.
	[Parameter(Mandatory = $True)]
	[string]$containerName,

	# Namespace of the repository/image. The microsoft in microsoft/iis.
	[Parameter(Mandatory = $True)]
	[string]$imageNamespace,

	# Name of the image to create. The iis in microsoft/iis.
	[Parameter(Mandatory = $True)]
	[string]$imageName,

	# Version/tag of the image to install. Defaults to latest.
	[Parameter(Mandatory = $True)]
	[string]$imageVersion,

	# URL of the registry. Leave empty to use Docker Hub.
	[Parameter(Mandatory = $False)]
	[string]$registryUrl,

	# Any extra parameters to add to the docker run command.
	[Parameter(Mandatory = $False)]
	[string]$runParameters,

	[Parameter()]
    [switch]$useLocalImage
)

$ErrorActionPreference = "Stop"

$myDirectoryPath = $PSScriptRoot

if (!$myDirectoryPath) {
	$myDirectoryPath = "."
}

if ((& (Join-Path $myDirectoryPath './Test-IsAdmin.ps1')) -ne $true)
{
	Write-Error "Access denied. You must run this script with elevated permissions."
}

if ($IsLinux)
{
	$docker = "/usr/bin/docker"
}
else
{
	$docker = Join-Path (Join-Path $env:ProgramFiles "docker") "docker.exe"
}

if (!(Test-Path -PathType Leaf $docker))
{
	Write-Error "Docker not found."
}

if ($registryUrl) {
	$fullName = "$registryUrl/$imageNamespace/$($imageName):$imageVersion"
} else {
	$fullName = "$imageNamespace/$($imageName):$imageVersion"
}

if ($useLocalImage) {
	# Check if it actually exists.
	& $docker inspect --type image $fullName 2>&1 | Out-Null

	if ($LASTEXITCODE -ne 0) {
		Write-Error "The image $fullName is not found locally."
	}
}
else {
	Write-Host "Downloading the image $fullName."
	& $docker pull $fullName
	
	if ($LASTEXITCODE -ne 0) {
		Write-Error "Docker pull failed with exit code $LASTEXITCODE"
	}	
}

Write-Host "Deleting any container with conflicting name (if exists)."
$ErrorActionPreference = "SilentlyContinue"
& $docker rm -f $containerName 2>&1 | Out-Null
$ErrorActionPreference = "Stop"

# Make the Docker client credentials available to the service.
$dockerDaemonConfigFolder = if ($IsLinux) { "/etc/docker" } else { Join-Path $env:ProgramData "docker\config" }
$clientCredentialsFolder = Join-Path $dockerDaemonConfigFolder "DockerClientCredentials"
if (Test-Path $clientCredentialsFolder) {
	if ($IsLinux) {
		$runParameters += " --volume $clientCredentialsFolder`:/DockerClientCredentials/"
	} else {
		$runParameters += " --volume $clientCredentialsFolder`:C:\DockerClientCredentials\"
	}
}

Write-Host "Starting $containerName."

# Limit memory usage (in MB) just as a safeguard against runaway infrastructure service resource usage.
$memoryLimit = "256m"

# The COMPONENT_HOST_NAME environment variable is set to let the component know the actual host name where container runs on. The components may want to log it, for example.
# The splitting here feels dangerous but so far we do not have issues with spaces soooo okay good enough.
& $docker run --name $containerName --memory $memoryLimit --detach --restart always --network axinom --env COMPONENT_HOST_NAME=$([Environment]::MachineName) $runParameters.Split(" ") $fullName

if ($LASTEXITCODE -ne 0)
{
	Write-Error "Docker run failed with exit code $LASTEXITCODE"
}