[CmdletBinding()]
param(
	# Array of module names to install from the package. If not provided, all modules are going to be installed.
	[Parameter(Mandatory = $false)]
	[array]$moduleNames
)

$ErrorActionPreference = "Stop"

$assetsFolder = Join-Path $PSScriptRoot "Assets"
$deliveryProcessToolingPath = Join-Path $PSScriptRoot "Assets/DeliveryProcess.Tooling"
$deployComponent = Join-Path $deliveryProcessToolingPath "Deploy-Component.ps1"

if ($moduleNames) {
	$moduleFolders = Get-ChildItem "$assetsFolder/Modules" -Directory | Where-Object Name -In $moduleNames
} else {
	$moduleFolders = Get-ChildItem "$assetsFolder/Modules" -Directory
}

$modules = @{}

$moduleFolders | ForEach-Object {
	$moduleSpec = & (Join-Path $_.FullName "Module.ps1")
	$moduleName = $_.Name
	$modules[$moduleName] = @($moduleSpec | ForEach-Object { $_  })
}

# Make sure the configuration has been generated (and is valid).
& (Join-Path $PSScriptRoot "Validate-Configuration.ps1") -moduleNames $moduleNames

# Check if all the images exist in the local system.
foreach ($moduleName in $modules.Keys) {
	foreach ($component in $modules[$moduleName]) {
		$image = $component.Image
		$imageFullName = "$($image.Registry)/$($image.Namespace)/$($image.Name):$($image.Version)"

		& docker inspect --type image $imageFullName | Out-Null
		
		if ($LASTEXITCODE -ne 0) {
			Write-Error "The image $imageFullName in module $moduleName does not exist in the local system."
		}
	}
}

foreach ($moduleName in $modules.Keys) {
	Write-Host "Deploying '$moduleName' module."

	$moduleFolder = Join-Path $assetsFolder "Modules/$moduleName"

	$transformedConfigurationRoot = Join-Path $moduleFolder "TransformedConfiguration"

	$preInstallStepsFile = Join-Path $moduleFolder "Execute-PreInstallSteps.ps1"

	if (Test-Path $preInstallStepsFile) {
		Write-Host "Executing pre-install steps."
		& $preInstallStepsFile -components $modules[$moduleName] -transformedConfigurationPath $transformedConfigurationRoot -toolingPath $deliveryProcessToolingPath
	}

	foreach ($component in $modules[$moduleName]) {
		$image = $component.Image

		if (!$component.Name) {
			Write-Verbose "Will not deploy image '$($image.Name)' because it lacks component specification."
			continue
		}

		$componentConfigurationPackagePath = ""

		if ($component.AssetsKey) {
			$componentConfigurationPackagePath = Join-Path $transformedConfigurationRoot $component.AssetsKey
			if (!(Test-Path $componentConfigurationPackagePath)) {
				$componentConfigurationPackagePath = ""
			}
		}

		Write-Host "Attempting to deploy component $($component.Name)."

		& $deployComponent -imageNamespace $image.Namespace `
						   -imageName $image.Name `
						   -imageVersion $image.Version `
						   -namespace $component.Namespace `
						   -name $component.name `
						   -configurationPackagePath $componentConfigurationPackagePath `
						   -dataVolumeNames $component.DataVolumes `
						   -hasChecks:$component.HasChecks `
						   -grpc:$component.EnableGrpc `
						   -httpBehavior "Allow" `
						   -quickPublish:$component.EnableQuickPublish `
						   -hostnames $component.Hostnames `
						   -server "localhost" `
						   -registryUrl $image.Registry `
						   -delayBetweenServers 0 `
						   -timeoutWaitingForGreenDashboard 0
	}

	$postInstallStepsFile = Join-Path $moduleFolder "Execute-PostInstallSteps.ps1"

	if (Test-Path $postInstallStepsFile) {
		Write-Host "Executing post-install steps."
		& $postInstallStepsFile -components $modules[$moduleName] -transformedConfigurationPath $transformedConfigurationRoot -toolingPath $deliveryProcessToolingPath
	}
}

# To save space, remove images that are not used.
Write-Host "Removing unused/obsolete Docker images."
& docker image prune --all --force


Write-Host "All components have been successfully deployed."