<#
    Configures Docker daemon to use TLS with client authentication.
#>

[CmdletBinding()]
Param(
    # Keep existing keys and certificates. Useful when updating Docker and you don't want to break existing clients using Docker.
    [Parameter(Mandatory = $False)]
    [switch]$keepExistingKeys,

    # The name that is to be added to the server certificate as the DNS name of the server.
    [Parameter(Mandatory = $False)]
    [string]$certificateSubjectName = [Environment]::MachineName
)

$ErrorActionPreference = "Stop"

$myDirectoryPath = $PSScriptRoot

if (!$myDirectoryPath) {
    $myDirectoryPath = "."
}

if ((& (Join-Path $myDirectoryPath './Test-IsAdmin.ps1')) -ne $true) {
    Write-Error "Access denied. You must run this script with elevated permissions."
}

. (Join-Path $myDirectoryPath "./Functions.ps1")

if (!$IsLinux) {
    $opensslDirectoryPath = Join-Path $myDirectoryPath "openssl"

    # Add OpenSSL to path in the script scope for easier usage.
    $env:Path += ";$opensslDirectoryPath"

    # OpenSSL expects a configuration file to exist. Providing the location via the environment variable.
    $env:OPENSSL_CONF = Join-Path $opensslDirectoryPath "openssl.cfg"
}

$daemonConfigFolder = Get-DockerDaemonConfigFolder

try {
    Push-Location
    
    Set-Location $daemonConfigFolder

    $daemonConfigFilePath = Join-Path $daemonConfigFolder "daemon.json"

    $daemonConfig = @{}
    (Get-Content $daemonConfigFilePath | ConvertFrom-Json).PSObject.properties | ForEach-Object { $daemonConfig[$_.Name] = $_.Value }

    $daemonConfig["tlsverify"] = $true
    $daemonConfig["tlscacert"] = Join-Path $daemonConfigFolder "ca.pem"
    $daemonConfig["tlscert"] = Join-Path $daemonConfigFolder "server-cert.pem"
    $daemonConfig["tlskey"] = Join-Path $daemonConfigFolder "server-key.pem"
    
    $daemonConfig | ConvertTo-Json | Out-FileWithoutBOM $daemonConfigFilePath

    if ($keepExistingKeys) {
        return
    }

    Write-Host "Generating keys and certificates for authentication. Using '$certificateSubjectName' as the certificate subject name."

    # There is no straightforward way to tell if OpenSSL commands failed because return codes and stderr are not used in a way one expects.
    # Therefore error handling is not done.

    # CA certificate and it's key.
    & openssl req -new -newkey rsa:4096 -days 1825 -nodes -x509 -subj "/CN=Docker-CA" -keyout ca-key.pem -out ca.pem

    # Server certificate and it's key.
    & openssl genrsa -out server-key.pem 4096

    & openssl req -subj "/CN=$certificateSubjectName" -sha256 -new -key server-key.pem -out server.csr

    $certificateExtensions = @"
    subjectAltName = DNS:$certificateSubjectName,DNS:localhost,IP:172.31.250.254,IP:127.0.0.1
    extendedKeyUsage = serverAuth
"@

    $certificateExtensions | Out-FileWithoutBOM "extfile.cnf"

    & openssl x509 -req -days 1825 -sha256 -in server.csr -CA ca.pem -CAkey ca-key.pem -CAcreateserial -out server-cert.pem -extfile extfile.cnf

    # Client certificate and it's key.
    & openssl genrsa -out key.pem 4096
    & openssl req -subj "/CN=Client certificate for $certificateSubjectName" -new -key key.pem -out client.csr

    "extendedKeyUsage = clientAuth" | Out-FileWithoutBOM "extfile.cnf"
    & openssl x509 -req -days 1825 -sha256 -in client.csr -CA ca.pem -CAkey ca-key.pem -CAcreateserial -out cert.pem -extfile extfile.cnf
    
    # Combine the client certificate and private key into a PKCS#12 PFX format because some clients may prefer that form as credentials (like infrastructure services).
    [Guid]::NewGuid().Guid | Out-FileWithoutBOM "pfx_key_password.txt"
    & openssl pkcs12 -export -inkey key.pem -in cert.pem -out cert.pfx -password file:pfx_key_password.txt

    # Set up a subfolder with client credentials.
    $clientCredentialsFolder = Join-Path $daemonConfigFolder "DockerClientCredentials"

    if (Test-Path -PathType Container $clientCredentialsFolder) {
        Remove-Item -Recurse -Force $clientCredentialsFolder
    }

    New-Item -ItemType Directory $clientCredentialsFolder | Out-Null

    $clientFiles = @("cert.pem", "key.pem", "cert.pfx", "pfx_key_password.txt")
    foreach ($file in $clientFiles) {
        Move-Item $file -Destination $clientCredentialsFolder
    }

    Copy-Item "ca.pem" -Destination $clientCredentialsFolder

    # Do some clean-up.
    Remove-Item extfile.cnf
    Remove-Item client.csr
    Remove-Item server.csr
} finally {
    Pop-Location
}
