[CmdletBinding()]
Param(
    # Specifies whether it would be possible to communicate with Docker without credentials and over an unencrypted channel.
    [Parameter(Mandatory = $False)]
    [switch]$noAuthentication,

    # Keep existing keys and certificates. Useful when updating Docker and you don't want to break existing clients using Docker.
    [Parameter(Mandatory = $False)]
    [switch]$keepExistingKeys
)

$ErrorActionPreference = "Stop"

$myDirectoryPath = $PSScriptRoot

if (!$myDirectoryPath) {
    $myDirectoryPath = "."
}

if ((& (Join-Path $myDirectoryPath './Test-IsAdmin.ps1')) -ne $true) {
    Write-Error "Access denied. You must run this script with elevated permissions."
}

. (Join-Path $myDirectoryPath "./Functions.ps1")

$setupDockerSecurityScriptPath = Join-Path $myDirectoryPath "Setup-DockerAuthentication.ps1"

$daemonConfigFolder = Get-DockerDaemonConfigFolder

# The install process is completely different for Linux and Windows.
if ($IsLinux) {
    if ((which docker) -ne $null) {
        Write-Host "Docker is already installed. The script will attempt to upgrade it."
        Write-Warning "Docker upgrades are only supported between ASDP versions with no breaking changes!"

        & systemctl stop docker

        if ($LASTEXITCODE -ne 0) {
            Write-Error "Failed to stop already installed Docker service."
        }
    }

    $ubuntuVersion = & bash -c 'echo $(. /etc/os-release && echo "$VERSION_CODENAME")'
	
	if ($ubuntuVersion -eq "bionic") {
		$debsFolder = Join-Path $myDirectoryPath "docker-for-ubuntu18"
	} elseif ($ubuntuVersion -eq "xenial") {
		$debsFolder = Join-Path $myDirectoryPath "docker-for-ubuntu16"
    } else {
        Write-Error "This distribution of Linux is not supported."
    }

    Write-Host "Installing Docker."

    $daemonConfigFilePath = Join-Path $daemonConfigFolder "daemon.json"

    # Delete the potentially existing daemon configuration file because incorrect configuration may result in installation failure.
    if (Test-Path $daemonConfigFilePath) {
        Remove-Item $daemonConfigFilePath
    }
    
    foreach ($packageName in @("containerd.io.deb", "docker-cli.deb", "docker.deb")) {
        $debPath = Join-Path $debsFolder $packageName
        if (!(Test-Path $debPath)) {
            Write-Error "Docker installation package is broken. Expected file not found: $debPath"
        }

        # It is okay if this fails (dependencies missing).
        & sudo dpkg -i $debPath
    
        # This will fixup all the dependencies and finish installing Docker.
        & sudo apt-get install --no-install-recommends --no-install-suggests -y -f
    
        if ($LASTEXITCODE -ne 0) {
            Write-Error "Failed to install Docker"
        }
    }

    Write-Host "Creating first run configuration."

    $dockerServiceOverrideDirectory = "/etc/systemd/system/docker.service.d"
    $dockerServiceOverrideFile = Join-Path $dockerServiceOverrideDirectory "override.conf"

    # Create the parent directory if it does not exist, otherwise Set-Content will fail.
    if (!(Test-Path -PathType Container $dockerServiceOverrideDirectory)) {
        New-Item -ItemType Directory $dockerServiceOverrideDirectory | Out-Null
    }

    Set-Content -Path $dockerServiceOverrideFile -Value @"
[Service]
ExecStart=
ExecStart=/usr/bin/dockerd
"@

    # We listen on any IP because Docker actually creates the networks when it starts up.
    # After it tries to bind to these networks... yes that's right.
	$daemonConfig = @{
		hosts = @("tcp://0.0.0.0:6740", "unix://")
		bridge = "none"
	}

    $daemonConfig | ConvertTo-Json | Out-FileWithoutBOM $daemonConfigFilePath

    if (!$noAuthentication) {
        & $setupDockerSecurityScriptPath -keepExistingKeys:$keepExistingKeys
    }

    Write-Host "Performing first run."

    & systemctl daemon-reload

    if ($LASTEXITCODE -ne 0) {
        Write-Error "Failed to reload Docker startup registration."
    }

    & systemctl restart docker

    if ($LASTEXITCODE -ne 0) {
        Write-Error "Failed to start Docker for the first run."
    }

    # Delete it in case we are upgrading. It is okay if this fails - means we are not upgrading.
    Write-Host "Removing existing container network (if it exists)."

	& docker network rm axinom
	
    Write-Host "Setting up container network."

    & docker network create --driver bridge --gateway 172.31.250.254 --subnet 172.31.250.0/24 axinom

    if ($LASTEXITCODE -ne 0) {
        Write-Error "Failed to set up container network."
    }

    # Ensure it binds to the new network it just created.
    Write-Host "Restarting Docker to finalize configuration."

    & systemctl restart docker

    if ($LASTEXITCODE -ne 0) {
        Write-Error "Failed to restart Docker with the final configuration."
    }
}
else {
    $dockerDirectoryPath = Join-Path "$env:ProgramFiles" "docker"
    $dockerdPath = Join-Path $dockerDirectoryPath "dockerd.exe"
    $dockerPath = Join-Path $dockerDirectoryPath "docker.exe"

    if (Test-Path -PathType Container $dockerDirectoryPath) {
        Write-Host "Docker is already installed. The script will attempt to upgrade it."
        Write-Warning "Docker upgrades are only supported between ASDP versions with no breaking changes!"

        $serviceName = "docker"
        if (Get-Service $serviceName -ErrorAction SilentlyContinue) {
            if ((Get-Service $serviceName).Status -eq "Running") {
                Stop-Service $serviceName | Out-Null
            }

            & $dockerdPath --unregister-service
        }

        Remove-Item -Recurse -Force $dockerDirectoryPath
    }

    $dockerZipFilename = "docker-for-windows/docker-windows.zip"
    $opensslZipFilename = "openssl-windows.zip"
    $dockerZipPath = Join-Path $myDirectoryPath $dockerZipFilename
    $opensslZipPath = Join-Path $myDirectoryPath $opensslZipFilename

    # Check that we have everything we need before starting the show.
    $requiredFiles = @($dockerZipPath, $opensslZipPath)
    foreach ($path in $requiredFiles) {
        if (!(Test-Path -PathType Leaf $path)) {
            Write-Error "The file $path must exist alongside the script!"
        }
    }

    Write-Host "Installing Docker service."

    # ZipFile comes from here.
    Add-Type -AssemblyName System.IO.Compression.FileSystem

    # Unzip docker to Program Files, creating "Program Files\docker".
    # ExtractToDirectory does not use PowerShell working directory,
    # it uses the .NET working directory which is entirely different!
    $dockerZipPath = Resolve-Path $dockerZipPath # To absolute path.
    [System.IO.Compression.ZipFile]::ExtractToDirectory($dockerZipPath, $env:ProgramFiles)

    Write-Host "Setting up OpenSSL."

    # Let's first remove the already existing output folder if it exists.
    $opensslDirectoryPath = Join-Path $myDirectoryPath "openssl"
    if (Test-Path -PathType Container $opensslDirectoryPath) {
        Remove-Item -Recurse -Force $opensslDirectoryPath
    }

    [System.IO.Compression.ZipFile]::ExtractToDirectory(($opensslZipPath | Resolve-Path), $myDirectoryPath)

    Write-Host "Configuring Docker."

    # Add docker to path (system scope) for later easy usage if it is not already there.
    $oldPath = [Environment]::GetEnvironmentVariable("Path", [EnvironmentVariableTarget]::Machine)
    if ($oldPath -notcontains $dockerDirectoryPath) {
        [Environment]::SetEnvironmentVariable("Path", $oldPath + ";" + $dockerDirectoryPath, [EnvironmentVariableTarget]::Machine)
    }

    # Write the Docker daemon configuration.
    if (!(Test-Path $daemonConfigFolder)) {
        New-Item -ItemType Directory -Force $daemonConfigFolder | Out-Null
    }

	$daemonConfig = @{
		hosts = @("tcp://0.0.0.0:6740", "npipe://")
		bridge = "none"
	}

    $daemonConfigFilePath = (Join-Path $daemonConfigFolder "daemon.json")
    $daemonConfig | ConvertTo-Json | Out-FileWithoutBOM $daemonConfigFilePath

    if (!$noAuthentication) {
        & $setupDockerSecurityScriptPath -keepExistingKeys:$keepExistingKeys
    }
	
    Write-Host "Registering Windows service."

    # Register Docker as a Windows Service.
    & $dockerdPath --register-service

    if ($LASTEXITCODE -ne 0) {
        Write-Error "Failed to register Windows service."
    }

    # Experiment: on some servers Docker is known to have startup failures. Perhaps delayed start helps?
    # This delays Docker startup by 2 minutes from server start, which should be plenty for any "initialization".
    Set-ItemProperty -Path "HKLM:\SYSTEM\CurrentControlSet\Services\docker\" -Name DelayedAutostart -Value 1

    Write-Host "Starting Docker."

    Start-Service Docker

    # The stderr write when a network does not exist causes automation to think the script has failed.
    # We cannot redirect stderr->stdout because it still keeps the error in the pipeline (nice, Microsoft!)
    # So okay, just see if we can find the network from the listing and do it the long way around.
    $existingNetworkListing = & $dockerPath network ls --filter "name=axinom" --filter "driver=nat" --quiet | Out-String

    if ($existingNetworkListing) {
        Write-Host "Removing existing container network."
        & $dockerPath network rm axinom
    }

    Write-Host "Setting up container network."

    & $dockerPath network create --driver nat --gateway 172.31.250.254 --subnet 172.31.250.0/24 axinom

    if ($LASTEXITCODE -ne 0) {
        Write-Error "Failed to set up container network."
    }
}

    Write-Host "Configuring firewall rules."

    if ($IsLinux) {
        # On Linux we not only have to allow port 6740 but actually ALL traffic because otherwise
        # ufw blocks container-container traffic when enabled in some (all?) scenarios.
        ufw allow from 172.31.250.0/24 to any
    
        if ((ufw status) -ne "Status: inactive") {
            ufw reload
        }
    }
    else {
        # On Windows, everything else is automatically
        Write-Host "Allowing Docker management traffic in firewall."
        Get-NetFirewallRule | Where-Object { $_.DisplayName -eq "Allow Docker management on internal network" } | Remove-NetFirewallRule
        New-NetFirewallRule -DisplayName "Allow Docker management on internal network" -Direction Inbound -LocalPort 6740 -Protocol TCP -Action Allow -RemoteAddress "172.31.250.0/24" | Out-Null
    }
    
    Write-Host "Docker setup completed."