$ErrorActionPreference = "Stop"

$myDirectoryPath = $PSScriptRoot

if (!$myDirectoryPath) {
    $myDirectoryPath = "."
}

if ((& (Join-Path $myDirectoryPath './Test-IsAdmin.ps1')) -ne $true) {
    Write-Error "Access denied. You must run this script with elevated permissions."
}

. (Join-Path $myDirectoryPath "./Functions.ps1")

if ($IsLinux) {
    # Linux.
    $dockerMaintenanceConfigFilename = "DockerMaintenance-Linux.config"
}
else {
    $dockerMaintenanceConfigFilename = "DockerMaintenance-Windows$(GetWindowsVersion).config"
}

$dataRootPath = "/ComponentHost/"

$httpGatewayConfigFilename = "HttpGateway.config"
$deploymentAgentConfigFilename = "DeploymentAgent.config"
$dashboardConfigFilename = "CheckDashboard.config"

$httpGatewayConfigSourcePath = Join-Path $myDirectoryPath $httpGatewayConfigFilename
$deploymentAgentConfigSourcePath = Join-Path $myDirectoryPath $deploymentAgentConfigFilename
$dashboardConfigSourcePath = Join-Path $myDirectoryPath $dashboardConfigFilename
$dockerMaintenanceConfigSourcePath = Join-Path $myDirectoryPath $dockerMaintenanceConfigFilename

# Check that we have everything we need before starting the show.
$requiredFiles = @(
    $httpGatewayConfigSourcePath,
    $deploymentAgentConfigSourcePath,
    $dashboardConfigSourcePath,
    $dockerMaintenanceConfigSourcePath
)

foreach ($path in $requiredFiles) {
    if (!(Test-Path -PathType Leaf $path)) {
        Write-Error "The file $path must exist alongside the script!"
    }
}

Write-Host "Creating required directories."

# First, create all the empty directories we will need, if they do not already exist.
$directories = @(
    (Join-Path $dataRootPath "Components/Data/"),
    (Join-Path $dataRootPath "Components/Config/"),
    (Join-Path $dataRootPath "Components/Logs/"),
    (Join-Path $dataRootPath "Infrastructure/Data/deployment-agent/"),
    (Join-Path $dataRootPath "Infrastructure/Data/http-gateway/"),
    (Join-Path $dataRootPath "Infrastructure/Data/acme/"),
    (Join-Path $dataRootPath "Infrastructure/Logs/deployment-agent/"),
    (Join-Path $dataRootPath "Infrastructure/Logs/http-gateway/"),
    (Join-Path $dataRootPath "Infrastructure/Logs/docker-stats-exporter/"),
    (Join-Path $dataRootPath "Infrastructure/Logs/dashboard/"),
    (Join-Path $dataRootPath "Infrastructure/Logs/dockermaintenance/"),
    (Join-Path $dataRootPath "Infrastructure/Logs/acme/"),
    (Join-Path $dataRootPath "Infrastructure/Config/deployment-agent/"),
    (Join-Path $dataRootPath "Infrastructure/Config/http-gateway/"),
    (Join-Path $dataRootPath "Infrastructure/Config/docker-stats-exporter/"),
    (Join-Path $dataRootPath "Infrastructure/Config/dashboard/"),
    (Join-Path $dataRootPath "Infrastructure/Config/dockermaintenance/"),
    (Join-Path $dataRootPath "Infrastructure/Config/acme/")
)

foreach ($path in $directories) {
    if (!(Test-Path -PathType Container $path)) {
        New-Item -ItemType Directory $path | Out-Null
    }
}

# Write the process version in a file for later reference.
Set-Content -Path (Join-Path $dataRootPath "ProcessVersion.txt") -Value "9.0.0-015306-eae7db0"

Write-Host "Creating HTTP gateway configuration file."
$httpGatewayConfigDestinationPath = Join-Path $dataRootPath "Infrastructure/Config/http-gateway/Extra.config"
Copy-Item -Force $httpGatewayConfigSourcePath $httpGatewayConfigDestinationPath | Out-Null

Write-Host "Creating deployment agent configuration file."
$deploymentAgentConfigDestinationPath = Join-Path $dataRootPath "Infrastructure/Config/deployment-agent/DeploymentAgent.config"
Copy-Item -Force $deploymentAgentConfigSourcePath $deploymentAgentConfigDestinationPath | Out-Null

Write-Host "Creating Docker maintenance utility configuration file."
$dockerMaintenanceConfigDestinationPath = Join-Path $dataRootPath "Infrastructure/Config/dockermaintenance/DockerMaintenance.config"
Copy-Item -Force $dockerMaintenanceConfigSourcePath $dockerMaintenanceConfigDestinationPath | Out-Null

Write-Host "Creating dashboard configuration file."
$dashboardConfigDestinationPath = Join-Path $dataRootPath "Infrastructure/Config/dashboard/CheckDashboard.config"
Copy-Item -Force $dashboardConfigSourcePath $dashboardConfigDestinationPath | Out-Null

Write-Host "Ensuring clean state."

$deploymentAgentDataPath = Join-Path $dataRootPath "Infrastructure/Data/deployment-agent/"
Get-ChildItem $deploymentAgentDataPath | Remove-Item -Force -Recurse

$gatewayDataPath = Join-Path $dataRootPath "Infrastructure/Data/http-gateway/"
Get-ChildItem $gatewayDataPath | Remove-Item -Force -Recurse

$gatewayConfigPath = Join-Path $dataRootPath "Infrastructure/Config/http-gateway/"
Get-ChildItem $gatewayConfigPath | Remove-Item -Force -Recurse

$acmeDataPath = Join-Path $dataRootPath "Infrastructure/Data/acme/"
Get-ChildItem $acmeDataPath | Remove-Item -Force -Recurse

$acmeConfigPath = Join-Path $dataRootPath "Infrastructure/Config/acme/"
Get-ChildItem $acmeConfigPath | Remove-Item -Force -Recurse


if ($IsLinux) {
    # No filesystem permissions required on Linux, as Docker runs as root.
}
else {
    Write-Host "Assigning filesystem permissions"

    # Grant the required filesystem permissions
    # "Users" gets read-write-modify.
    # "Administrators" and "SYSTEM" get full control.

    # If dataRootPath is a junction then apply the permisisons on the target instead.
    $dataStoragePath = $dataRootPath

    if ((Get-Item $dataRootPath).LinkType -ne $null) {
        # It is a junction.
        $dataStoragePath = (Get-Item $dataRootPath).Target
    }

    $acl = Get-Acl -Path $dataStoragePath

    $usersRule = New-Object -TypeName System.Security.AccessControl.FileSystemAccessRule -ArgumentList "Users", "Modify", ("ContainerInherit, ObjectInherit"), "None", "Allow"
    $administratorsRule = New-Object -TypeName System.Security.AccessControl.FileSystemAccessRule -ArgumentList "Administrators", "FullControl", ("ContainerInherit, ObjectInherit"), "None", "Allow"
    $systemRule = New-Object -TypeName System.Security.AccessControl.FileSystemAccessRule -ArgumentList "NT Authority\System", "FullControl", ("ContainerInherit, ObjectInherit"), "None", "Allow"

    $acl.SetAccessRule($usersRule)
    $acl.SetAccessRule($administratorsRule)
    $acl.SetAccessRule($systemRule)

    Set-Acl -Path $dataStoragePath -AclObject $acl
}

Write-Host "Success."